      Program mod2ci
! ----------------------------------------------------------------------
! **********************************************************************
! *** This program is part of the Euroben Benchmark                  ***
! ***                                                                ***
! *** Copyright: EuroBen Group p/o                                   ***
! ***            Utrecht University, Computational Physics Dept.     ***
! ***            P.O. Box 80.000                                     ***
! ***            3508 TA Utrecht                                     ***
! ***            The Netherlands                                     ***
! ***                                                                ***
! *** Author of this program: Aad van der Steen                      ***
! *** Date                    Summer 2005                            ***
! **********************************************************************
! ----------------------------------------------------------------------
! --- Version 1.0 (Distributed memory, parallel, MPI).

! --- Purpose of program mod2ci
! -----------------------------
! --- Solver for sparse linear systems with iterative methods and
!     one type of preconditioner. The systems are not actually
!     solved but rather a predifined number of iterations is performed
!     to assess the speed of the solver-preconditioner combination.
!     Two types of systems are considered:
!     1. - Systems stemming from 3-D finite difference problems
!          resulting in 7-banded matrices. We consider only the
!          symmetric type in ANOTHER program: 'mod2cr'.
!     2. - Finite element type, irregularly filled matrices.
!          They are stored in CRS format. In this program 'mod2ci' we
!          only look at non-symmetric matrices and their solution
!          methods.
!
! --- In this Program 'mod2ci' we address the irregular, FEM type
!     of problem. For the regular 7-band matrix-based type we have
!     program 'mod2cr'.
!
!     Solvers used:
!     -  For the symmetric banded systems in 'mod2cr': CG with ILU(0)
!        and polynomial preconditioning.
!     -  For irregular systems in 'mod2ci': RGMRES and TFQMR with
!        polynomial preconditioning.
! ----------------------------------------------------------------------
      Use                       numerics
      Use                       floptime
      Use                       mpi_module
      Implicit                  None

      Integer, Allocatable   :: indx(:), rowp(:)
      Real(l_), Allocatable  :: matvals(:), b(:), q(:), x(:)
      Real(l_)               :: corr, maxdf, mindf, res
      Real(l_)               :: frac, gtime, mflops, resnrm, time
      Integer, Parameter     :: maxit = 50
      Integer, Parameter     :: mp = 1
      Real(l_), Parameter    :: tol = 1.0e-10_l_
      Real(l_), Parameter    :: micro = 1.0e-6_l_, two = 2.0_l_,
     &                          perc = 1.0e2_l_
      Real(l_)               :: gamma(mp+1)
      Integer                :: i, lelmts, lrows, ncols, nelmts, nrows,
     &                          nrep
      Integer(8)             :: gflops
      Integer                :: rg
      Logical                :: ok
      External               :: lpolyn
! ----------------------------------------------------------------------
      Call mpistart
      If ( me == 0 ) Call state( 'mod2ci  ' )
      Open( 1, File = 'mod2ci.in' )
      If ( me == 0 ) Print 1000, nodes
! ----------------------------------------------------------------------
! --- Do case with GMRES and polynomial preconditioning.
 
      If ( me == 0 ) Print 1005
   10 Read( 1, *, End = 20 ) ncols, nrows, nelmts, nrep
         Call evdist( nrows )
         Call bsaddr
         lrows  = part( nrows )
         lelmts = part( nelmts )     
         Allocate( indx(lelmts), rowp(lrows+1), matvals(lelmts),
     &             b(ncols), q(nrows), x(nrows) )
         Call getmatvec( ncols, lrows, lelmts, indx, rowp, matvals, b )
         Call pcoefs( mp, gamma )
         q = 1.0_l_
         flops = 0
         time = MPI_Wtime()
         Do i = 1, nrep
            x = 0.0_l_
            resnrm = -1.0_l_
            Call rgmres( ncols, lrows, lelmts, mp, indx, rowp, matvals,
     &                   q, x, b, gamma, maxit, tol, resnrm, lpolyn )
         End Do
         time = MPI_Wtime() - time
         Call MPI_Allreduce(flops, gflops, 1, ityp, MPI_Sum, comm, ierr)
         Call MPI_Allreduce( time, gtime, 1, rtyp, MPI_Max, comm, ierr )
         mflops = micro*Real( gflops, l_ )/time
         gtime   = gtime/Real( nrep, l_ )
         frac   = perc*Real( nelmts, l_ )/( Real( ncols, l_)*
     &                 Real( nrows, l_ ) )
         If ( me == 0 ) Print 1010, ncols, nrows, frac, gtime, mflops,
     &                              resnrm
         Deallocate( indx, rowp, matvals, b, q, x )
      Go To 10
   20 If ( me == 0 ) Print 1020
      If ( me == 0 ) Print 1025
! ----------------------------------------------------------------------
! --- Do case with TFQMR and polynomial preconditioning.
 
      Rewind 1
      If ( me == 0 ) Print 1030, nodes
      If ( me == 0 ) Print 1005
   30 Read( 1, *, End = 40 ) ncols, nrows, nelmts, nrep
         Call evdist( nrows )
         Call bsaddr
         lrows  = part( nrows )
         lelmts = part( nelmts )     
         Allocate( indx(lelmts), rowp(lrows+1), matvals(lelmts),
     &             b(ncols), q(nrows), x(nrows) )
         Call getmatvec( ncols, lrows, lelmts, indx, rowp, matvals, b )
         Call pcoefs( mp, gamma )
         q = 1.0_l_
         flops = 0
         time = MPI_Wtime()
         Do i = 1, nrep
            x = 0.0_l_
            resnrm = -1.0_l_
            Call tfqmr( ncols, lrows, lelmts, mp, indx, rowp, matvals,
     &                  q, x, b, gamma, maxit, tol, resnrm, lpolyn )
         End Do
         time = MPI_Wtime() - time
         Call MPI_Allreduce(flops, gflops, 1, ityp, MPI_Sum, comm, ierr)
         Call MPI_Allreduce( time, gtime, 1, rtyp, MPI_Max, comm, ierr )
         mflops = micro*Real( gflops, l_ )/time
         gtime  = gtime/Real( nrep, l_ )
         frac   = perc*Real( nelmts, l_ )/( Real( ncols, l_)*
     &                 Real( nrows, l_ ) )
         If ( me == 0 ) Print 1010, ncols, nrows, frac, gtime, mflops,
     &                              resnrm
         Deallocate( indx, rowp, matvals, b, q, x )
      Go To 30
   40 If ( me == 0 ) Print 1020
      Call mpibye
! ----------------------------------------------------------------------
 1000 Format( 'Program mod2ci: Sparse iterative solver test'/
     &        'Non-symmetric, CRS: RGMRES with polyn. preconditioner.'/
     &        'No. of proc.s = ', i5 )
 1005 Format( '-------------------------------------------------------',
     &        '-----------'/
     &        ' #Rows | #Cols | %Fill |   Time(s)   |   Mflop/s   |   R'
     &        'esidue   |'/
     &        '-------------------------------------------------------',
     &        '----------|' )
 1010 Format( i7, '|', i7, '|', f6.2, ' |', g13.5, '|', g13.5, '|',
     &        g13.5, '|' )
 1020 Format( '-------------------------------------------------------', 
     &        '-----------' )
 1025 Format ( / )
 1030 Format( '-------------------------------------------------------',
     &        '-----------'/
     &        'Program mod2ci: Sparse iterative solver test'/
     &        'Non-symmetric, CRS: TFQMR with polyn. preconditioner.'/
     &        'No. of proc.s = ', i5 )
! ----------------------------------------------------------------------
      End Program mod2ci
