#include "dfmaxmin.h"

int dtrsm( char *side, char *uplo, char *transa, char *diag, 
	   int m, int n, double alpha, double *a, int lda, double *b, int ldb )
{
    extern int lsame( char *, char * );
    extern int xerbla( char *, int );

    /* Local variables */
    int i, j, k, info;
    int mtest, rtest;
    double temp;
    int lside, nrowa, upper, nounit;

/*  Purpose
    =======
    DTRSM  solves one of the matrix equations

       op( A )*X = alpha*B,   or   X*op( A ) = alpha*B,

    where alpha is a scalar, X and B are m by n matrices, A is a unit, or
    non-unit,  upper or lower triangular matrix  and  op( A )  is one  of

       op( A ) = A   or   op( A ) = A'.

    The matrix X is overwritten on B.

    Arguments
    ==========
    SIDE   - * char.
             On entry, SIDE specifies whether op( A ) appears on the left
             or right of X as follows:
                SIDE = 'L' or 'l'   op( A )*X = alpha*B.
                SIDE = 'R' or 'r'   X*op( A ) = alpha*B.
             Unchanged on exit.
    UPLO   - * char.
             On entry, UPLO specifies whether the matrix A is an upper or
             lower triangular matrix as follows:
                UPLO = 'U' or 'u'   A is an upper triangular matrix.
                UPLO = 'L' or 'l'   A is a lower triangular matrix.
             Unchanged on exit.
    TRANSA - * char.
             On entry, TRANSA specifies the form of op( A ) to be used in
             the matrix multiplication as follows:
                TRANSA = 'N' or 'n'   op( A ) = A.
                TRANSA = 'T' or 't'   op( A ) = A'.
                TRANSA = 'C' or 'c'   op( A ) = A'.
             Unchanged on exit.
    DIAG   - * char.
             On entry, DIAG specifies whether or not A is unit triangular
             as follows:
                DIAG = 'U' or 'u'   A is assumed to be unit triangular.
                DIAG = 'N' or 'n'   A is not assumed to be unit triangular.
             Unchanged on exit.
    M      - int.
             On entry, M specifies the number of rows of B. M must be at
             least zero.
             Unchanged on exit.
    N      - int.
             On entry, N specifies the number of columns of B.  N must be
             at least zero.
             Unchanged on exit.
    ALPHA  - double.
             On entry, ALPHA specifies the scalar  alpha. When ALPHA is
             zero then A is not referenced and B need not be set before
             entry.
             Unchanged on exit.
    A      - double * of DIMENSION ( LDA*k ), where k is m
             when  SIDE = 'L' or 'l'  and is n  when  SIDE = 'R' or 'r'.
             Before entry  with  UPLO = 'U' or 'u', the leading  k by k
             upper triangular part of the array  A must contain the upper
             triangular matrix  and the strictly lower triangular part of
             A is not referenced.
             Before entry  with  UPLO = 'L' or 'l', the leading k by k
             lower triangular part of the array A must contain the lower
             triangular matrix  and the strictly upper triangular part of
             A is not referenced.
             Note that when  DIAG = 'U' or 'u',  the diagonal elements of
             A  are not referenced either, but are assumed to be  unity.
             Unchanged on exit.
    LDA    - int.
             On entry, LDA specifies the first dimension of A as declared
             in the calling (sub) program.  When SIDE = 'L' or 'l' then
             LDA  must be at least  max( 1, m ), when SIDE = 'R' or 'r'
             then LDA must be at least max( 1, n ).
             Unchanged on exit.
    B      - double * of DIMENSION ( LDB*n ).
             Before entry, the leading m by n part of the array B must
             contain the right-hand side matrix B, and on exit is
             overwritten by the solution matrix X.
    LDB    - int.
             On entry, LDB specifies the first dimension of B as declared
             in the calling (sub)program. LDB must be at least max( 1, m ).
             Unchanged on exit.

   -- Level 3 Blas routine. */
/* -- Written on 8-February-1989.
      Jack Dongarra, Argonne National Laboratory.
      Iain Duff, AERE Harwell.
      Jeremy Du Croz, Numerical Algorithms Group Ltd.
      Sven Hammarling, Numerical Algorithms Group Ltd. */

// -- Function body: Test the input parameters. */

   lside = lsame( side, "L" );
   if ( lside ) nrowa = m;
   else nrowa = n;
   nounit = lsame( diag, "N" );
   upper  = lsame( uplo, "U" );
   info = 0;
   rtest = max( 1, nrowa );
   mtest = max( 1, n );
   if ( !lside && !lsame( side, "R" ) ) info = 1;
   else if ( !upper && !lsame( uplo, "L" ) ) info = 2;
   else if ( !lsame( transa, "N" ) && !lsame( transa, "T" ) &&
             !lsame( transa, "C" ) ) info = 3;
   else if ( !lsame( diag, "U" ) && !lsame( diag, "N" ) ) info = 4;
   else if ( m < 0 ) info = 5;
   else if ( n < 0 ) info = 6;
   else if ( lda < rtest ) info = 9;
   else if ( ldb < mtest ) info = 11;
   if ( info != 0 ) {
      xerbla( "DTRSM ", info );
      return 0;
   }

// -- Quick return if possible. 

   if ( n == 0 ) return 0;

// -- And when alpha == 0.0.

   if ( alpha == 0.0 ) {
      for( j = 0; j < m; ++j ) {
	 for( i = 0; i <= n; ++i ) {
            b[j*ldb+i] = 0.0;
         }
      }
      return 0;
   }

// -- Start the operations. 
   if ( lside ) {
      if ( lsame( transa, "N" ) ) {

// --    Form  B := alpha*inv( A )*B. 
         if ( upper ) {
            for( j = 0; j < m; ++j ) {
               if ( alpha != 1.0 ) {
                  for(i = 0; i < n; ++i ) {
                     b[j*ldb+i] *= alpha;
                  }
               }
            }
            for( j = 0; j < n; ++j ) {
               for( k = m-1; k >= 0; --k ) {
                  if ( b[k*ldb+j] != 0.0 ) {
                     if ( nounit ) b[k*ldb+j] /= a[k*lda+k];
                     for(i = 0; i < k; ++i ) {
                        b[i*ldb+j] -= b[k*ldb+j]*a[i*lda+k];
                     }
                  }
               }
            }
         }
         else {
            for( j = 0; j < m; ++j ) {
               if ( alpha != 1.0 ) {
                  for(i = 0; i < n; ++i ) {
                     b[i*ldb+i] *= alpha;
                  }
               }
            }
            for( j = 0; j < n; ++j ) {
               for( k = 0; k < m; ++k ) {
                  if ( b[k*ldb+j] != 0.0 ) {
                     if ( nounit ) b[k*ldb+j] /= a[k*lda+k];
                     for( i = k+1; i < m; ++i ) {
                        b[i*ldb+j] -= b[k*ldb+j]*a[i*lda+k];
                     }
                  }
               }
            }
         }
      }
      else {

// --    Form  B := alpha*inv( A' )*B. 
         if ( upper ) {
            for( j = 0; j < n; ++j ) {
               for( i = 0; i < m; ++i ) {
                  temp = alpha*b[i*ldb+j];
                  for( k = 0; k < i-1; ++k ) {
                     temp -= a[k*lda+i]*b[k*ldb+j];
                  }
                  if ( nounit ) temp /= a[i*lda+i];
                  b[i*ldb+j] = temp;
               }
            }
         }
         else {
            for( j = 0; j < n; ++j ) {
               for( i = m-1; i >= 0; --i ) {
                  temp = alpha*b[i*ldb+j];
                  for( k = i+1; k < m; ++k ) {
                     temp -= a[k*lda+i]*b[k*ldb+j];
                  }
                  if ( nounit ) temp /= a[i*lda+i];
                  b[i*ldb+j] = temp;
               }
            }
         }
      }
   }
   else {
      if ( lsame( transa, "N" ) ) {

// --    Form  B := alpha*B*inv( A ). 
         if ( upper ) {
            for( j = 0; j < m; ++j ) {
               if ( alpha != 1.0 ) {
                  for( i = 0; i < n; ++i ) {
                     b[i*ldb+j] *= alpha;
                  }
               }
            }
            for( j = 0; j < n; ++j ) {
               for( k = 0; k < j-1; ++k ) {
                  if ( a[k*lda+j] != 0.0 ) {
                     for( i = 0; i < m; ++i ) {
                        b[i*ldb+j] -= a[k*lda+j]*b[i*ldb+k];
                     }
                  }
               }
               if ( nounit ) {
                  temp = 1.0/a[j*lda+j];
                  for( i = 0; i < m; ++i ) {
                     b[i*ldb+j] *= temp;
                  }
               }
            }
         }
         else {
            for( i = 0; i < m; ++i ) {
               if ( alpha != 1.0 ) {
                  for( j = n-1; j >= 0; --j ) {
                     b[i*ldb+j] *= alpha;
                  }
               }
            }
            for( j = n-1; j >= 0; --j ) {
               for( k = j+1; k < n; ++k ) {
                  if ( a[k*lda+j] != 0.0 ) {
                     for( i = 0; i < m; ++i ) {
                        b[i*ldb+j] -= a[k*lda+j]*b[i*ldb+k];
                     }
                  }
               }
               if ( nounit ) {
                  temp = 1.0/a[j*lda+j];
                  for( i = 0; i < m; ++i ) {
                     b[i*ldb+j] *= temp;
                  }
               }
            }
         }
      }
      else {

// --    Form  B := alpha*B*inv( A' ). 
         if ( upper ) {
            for( i = 1; i < m; ++i ) {
               if ( nounit ) {
                  temp = 1.0/a[k*lda+k];
                  for( k = n-1; k >= 0; --k ) {
                     b[i*ldb+k] *= temp;
                  }
               }
            }
            for( k = n-1; k >= 0; --k ) {
               for( j = 0; j < k-1; ++j ) {
                  if ( a[j*lda+k] != 0.0 ) {
                     temp = a[j*lda+k];
                     for( i = 0; i < m; ++i ) {
                        b[i*ldb+j] -= temp*b[i*ldb+k];
                     }
                  }
               }
               if ( alpha != 1.0 ) {
                  for( i = 0; i < m; ++i ) {
                     b[i*ldb+k] *= alpha;
                  }
               }
            }
         }
         else {
            for( k = 0; k < n; ++k ) {
               if ( nounit ) {
                  temp = 1.0/a[k*lda+k];
                  for( i = 0; i < m; ++i ) {
                     b[i*ldb+k] *= temp;
                  }
               }
               for( j = k+1; j < n; ++j ) {
                  if ( a[j*lda+k] != 0.0 ) {
                     temp = a[j*lda+k];
                     for( i = 0; i < m; ++i ) {
                        b[i*ldb+j] -= temp*b[i*ldb+k];
                     }
                  }
               }
               if ( alpha != 1.0 ) {
                  for( i = 0; i < m; ++i ) {
                     b[i*ldb+k] *= alpha;
                  }
               }
            }
         }
      }
   }
   return 0;
} // -- End of dtrsm
